<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class SimpleCurrencyService
{
    private $cacheKey = 'currency_rates';
    private $cacheDuration = 3600; // 1 saat

    public function getRates()
    {
        // SORGU PASİF - Sadece varsayılan değerleri döndür
        return $this->getDefaultRates();
        
        /* PASİF EDİLEN KOD - API sorgusu yapılmıyor
        try {
            // Cache'den kontrol et
            $cachedRates = Cache::get($this->cacheKey);
            if ($cachedRates) {
                return $cachedRates;
            }

            // TCMB API'den kur bilgilerini çek
            $response = Http::timeout(10)->get('https://www.tcmb.gov.tr/kurlar/today.xml');
            
            if ($response->successful()) {
                $xml = simplexml_load_string($response->body());
                $rates = $this->parseXmlRates($xml);
                
                // Cache'e kaydet
                Cache::put($this->cacheKey, $rates, $this->cacheDuration);
                
                return $rates;
            }
        } catch (\Exception $e) {
            Log::error('Currency API Error: ' . $e->getMessage());
        }

        // Hata durumunda varsayılan değerler döndür
        return $this->getDefaultRates();
        */
    }

    private function parseXmlRates($xml)
    {
        $rates = [];
        
        if (isset($xml->Currency)) {
            foreach ($xml->Currency as $currency) {
                $code = (string)$currency['CurrencyCode'];
                $name = (string)$currency->CurrencyName;
                $buying = (string)$currency->BanknoteBuying;
                $selling = (string)$currency->BanknoteSelling;
                
                // Sadece istediğimiz kurları al
                if (in_array($code, ['USD', 'EUR', 'GBP'])) {
                    $rates[$code] = [
                        'name' => $name,
                        'buying' => $buying,
                        'selling' => $selling,
                        'code' => $code
                    ];
                }
            }
        }
        
        return $rates;
    }

    private function getDefaultRates()
    {
        return [
            'USD' => [
                'name' => 'US Dollar',
                'buying' => '39.90',
                'selling' => '40.10',
                'code' => 'USD'
            ],
            'EUR' => [
                'name' => 'Euro',
                'buying' => '46.80',
                'selling' => '47.00',
                'code' => 'EUR'
            ],
            'GBP' => [
                'name' => 'British Pound',
                'buying' => '54.70',
                'selling' => '55.00',
                'code' => 'GBP'
            ]
        ];
    }

    public function getFormattedRates()
    {
        $rates = $this->getRates();
        $formatted = [];
        
        foreach ($rates as $code => $rate) {
            $formatted[$code] = [
                'name' => $rate['name'],
                'buying' => number_format((float)$rate['buying'], 4, ',', '.'),
                'selling' => number_format((float)$rate['selling'], 4, ',', '.'),
                'code' => $rate['code']
            ];
        }
        
        return $formatted;
    }
}
