<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Google\Client;
use Google\Service\AnalyticsReporting;

class GoogleReportController extends Controller
{
    public function getReferers(Request $request)
    {
        $client = $this->getGoogleClient();

        if ($request->has('code')) {
            $this->authenticateClient($client, $request->input('code'));
            return redirect()->route('analytics.referers');
        }

        if ($client->getAccessToken() && !$client->isAccessTokenExpired()) {
            return $this->fetchReferersData($client);
        }

        return redirect($client->createAuthUrl());
    }

    private function getGoogleClient()
    {
        $client = new Client();
        $client->setApplicationName(config('google.application_name'));
        $client->setClientId(config('google.client_id'));
        $client->setClientSecret(config('google.client_secret'));
        $client->setRedirectUri(config('google.redirect_uri'));
        $client->setScopes(config('google.scopes'));
        $client->setAccessType(config('google.access_type'));
        $client->setApprovalPrompt(config('google.approval_prompt'));

        if (file_exists(config('google.credentials_path'))) {
            $accessToken = json_decode(file_get_contents(config('google.credentials_path')), true);
            $client->setAccessToken($accessToken);

            if ($client->isAccessTokenExpired()) {
                $newToken = $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
                $client->setAccessToken($newToken);
                file_put_contents(config('google.credentials_path'), json_encode($client->getAccessToken()));
            }
        }

        return $client;
    }

    private function authenticateClient(Client $client, $authCode)
    {
        $accessToken = $client->fetchAccessTokenWithAuthCode($authCode);
        if (isset($accessToken['error'])) {
            throw new \Exception(join(', ', $accessToken));
        }
        $client->setAccessToken($accessToken);
        file_put_contents(config('google.credentials_path'), json_encode($accessToken));
    }

    private function fetchReferersData(Client $client)
    {
        $analytics = new AnalyticsReporting($client);

        $dateRange = new \Google_Service_AnalyticsReporting_DateRange();
        $dateRange->setStartDate('30daysAgo');
        $dateRange->setEndDate('today');

        $sessions = new \Google_Service_AnalyticsReporting_Metric();
        $sessions->setExpression('ga:sessions');
        $sessions->setAlias('sessions');

        $referer = new \Google_Service_AnalyticsReporting_Dimension();
        $referer->setName('ga:source');

        $request = new \Google_Service_AnalyticsReporting_ReportRequest();
        $request->setViewId('YOUR_VIEW_ID');
        $request->setDateRanges($dateRange);
        $request->setMetrics([$sessions]);
        $request->setDimensions([$referer]);

        $body = new \Google_Service_AnalyticsReporting_GetReportsRequest();
        $body->setReportRequests([$request]);

        $reports = $analytics->reports->batchGet($body);

        $referers = [];
        foreach ($reports->getReports()[0]->getData()->getRows() as $row) {
            $referers[] = [
                'source' => $row->getDimensions()[0],
                'sessions' => $row->getMetrics()[0]->getValues()[0]
            ];
        }

        return view('analytics.referers', compact('referers'));
    }
}
